<?php
require_once '../config/config.php';
session_start();

header('Content-Type: application/json');

if (!isset($_SESSION['user_id'])) {
    http_response_code(401);
    echo json_encode(['error' => 'Unauthorized']);
    exit;
}

function generateSerialNumber($prefix = '', $length = 8) {
    $characters = '0123456789ABCDEFGHIJKLMNOPQRSTUVWXYZ';
    $serial = $prefix;
    
    for ($i = 0; $i < $length; $i++) {
        $serial .= $characters[rand(0, strlen($characters) - 1)];
    }
    
    return $serial;
}

function validateSerialNumbers($serial_numbers, $quantity) {
    // Remove empty values and trim whitespace
    $serial_numbers = array_filter(array_map('trim', $serial_numbers));
    
    // Check if number of serials matches quantity
    if (count($serial_numbers) !== $quantity) {
        return false;
    }
    
    // Check for duplicates
    if (count($serial_numbers) !== count(array_unique($serial_numbers))) {
        return false;
    }
    
    return true;
}

try {
    $action = $_POST['action'] ?? '';
    $quantity = intval($_POST['quantity'] ?? 0);
    $prefix = 'SN'; // Always use SN prefix

    if ($quantity <= 0) {
        throw new Exception('Invalid quantity');
    }

    switch ($action) {
        case 'generate':
            $serials = [];
            // Get the last serial number for this product
            $stmt = $pdo->prepare("
                SELECT MAX(CAST(SUBSTRING(serial_number, 3) AS UNSIGNED)) as last_number 
                FROM product_serial_numbers 
                WHERE serial_number LIKE 'SN%'
            ");
            $stmt->execute();
            $lastNumber = $stmt->fetchColumn() ?: 0;

            // Generate sequential numbers
            for ($i = 1; $i <= $quantity; $i++) {
                $nextNumber = $lastNumber + $i;
                $serials[] = $prefix . str_pad($nextNumber, 5, '0', STR_PAD_LEFT);
            }
            echo json_encode(['serials' => $serials]);
            break;

        case 'validate':
            $serial_numbers = json_decode($_POST['serial_numbers'] ?? '[]', true);
            $quantity = intval($_POST['quantity'] ?? 0);
            
            if (empty($serial_numbers) || $quantity <= 0) {
                throw new Exception('Serial numbers and quantity are required');
            }

            if (count($serial_numbers) !== $quantity) {
                throw new Exception("Number of serial numbers (".count($serial_numbers).") must match the quantity ($quantity)");
            }

            // Check for duplicates in the input
            if (count($serial_numbers) !== count(array_unique($serial_numbers))) {
                throw new Exception('Duplicate serial numbers are not allowed');
            }

            // Check against database for existing serial numbers
            $placeholders = str_repeat('?,', count($serial_numbers) - 1) . '?';
            $stmt = $pdo->prepare("
                SELECT serial_number 
                FROM product_serial_numbers 
                WHERE serial_number IN ($placeholders)
            ");
            $stmt->execute($serial_numbers);
            $existing_serials = $stmt->fetchAll(PDO::FETCH_COLUMN);

            if (!empty($existing_serials)) {
                throw new Exception('The following serial numbers already exist: ' . implode(', ', $existing_serials));
            }

            echo json_encode([
                'valid' => true,
                'message' => 'Serial numbers validated successfully'
            ]);
            break;

        case 'save':
            $product_id = intval($_POST['product_id'] ?? 0);
            $serial_numbers = $_POST['serial_numbers'] ?? [];
            $quantity = intval($_POST['quantity'] ?? 0);
            
            if ($product_id <= 0 || empty($serial_numbers) || $quantity <= 0) {
                throw new Exception('Invalid input');
            }

            // Verify product exists
            $stmt = $pdo->prepare("SELECT id FROM products WHERE id = ?");
            $stmt->execute([$product_id]);
            if (!$stmt->fetch()) {
                throw new Exception('Invalid product ID');
            }
            
            try {
                $pdo->beginTransaction();
                
                // Validate and format each serial number
                $formatted_serials = [];
                foreach ($serial_numbers as $serial) {
                    // Ensure serial number starts with SN and has proper padding
                    if (!str_starts_with($serial, 'SN')) {
                        $serial = 'SN' . str_pad($serial, 5, '0', STR_PAD_LEFT);
                    }
                    
                    // Check for duplicates
                    $stmt = $pdo->prepare("SELECT COUNT(*) FROM product_serial_numbers WHERE serial_number = ?");
                    $stmt->execute([$serial]);
                    if ($stmt->fetchColumn() > 0) {
                        throw new Exception("Serial number $serial already exists");
                    }
                    
                    $formatted_serials[] = $serial;
                }

                // Insert serial numbers
                $stmt = $pdo->prepare("
                    INSERT INTO product_serial_numbers 
                    (product_id, serial_number, status, source_type, source_id)
                    VALUES (?, ?, 'available', 'manual_adjustment', ?)
                ");
                
                foreach ($formatted_serials as $serial) {
                    $stmt->execute([$product_id, $serial, $_SESSION['user_id']]);
                }

                // Verify all serials were inserted
                $placeholders = str_repeat('?,', count($formatted_serials) - 1) . '?';
                $stmt = $pdo->prepare("
                    SELECT COUNT(*) FROM product_serial_numbers 
                    WHERE product_id = ? AND serial_number IN ($placeholders)
                ");
                $stmt->execute(array_merge([$product_id], $formatted_serials));
                if ($stmt->fetchColumn() !== count($formatted_serials)) {
                    throw new Exception('Failed to save all serial numbers');
                }
                
                // Update product quantity
                $updateStmt = $pdo->prepare("
                    UPDATE products 
                    SET quantity = quantity + ? 
                    WHERE id = ?
                ");
                $updateStmt->execute([$quantity, $product_id]);
                
                // Log the adjustment
                $logStmt = $pdo->prepare("
                    INSERT INTO stock_adjustments 
                    (product_id, quantity, reason, adjusted_by, adjusted_at)
                    VALUES (?, ?, 'Manual stock adjustment with serial numbers', ?, NOW())
                ");
                $logStmt->execute([$product_id, $quantity, $_SESSION['user_id']]);
                
                $pdo->commit();
                echo json_encode(['success' => true]);
                
            } catch (Exception $e) {
                if ($pdo->inTransaction()) {
                    $pdo->rollBack();
                }
                throw $e;
            }
            break;
            
        default:
            throw new Exception('Invalid action');
    }
} catch (Exception $e) {
    http_response_code(400);
    echo json_encode(['error' => $e->getMessage()]);
} 